<?php
/**
 * This file defines a class to abstract The Weather Channel's
 *  weather content for the front-end of the eWeather component.
 *
 * This file is part of eWeather.
 *
 *   eWeather is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   eWeather is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with eWeather.  If not, see <a href="http://www.gnu.org/licenses/">
 *   http://www.gnu.org/licenses/</a>.
 *
 * <b>Modifications:</b><br/>
 * Initial version by Bob Lavey
 *
 * @package eWeather
 * @subpackage classes
 * @copyright 2009 - 2011 Bob Lavey<br/>
 * @license http://www.gnu.org/licenses/gpl.txt GNU/GPL
 */

/* ensure this file is being called by Joomla! */
defined( '_JEXEC' ) or die( 'Direct Access to this location is not allowed.' );

// setup paths
require_once( JPATH_SITE.DS.'components'.DS.'com_eweather'.DS.'helpers'.DS.'defines.php' );
require_once( COM_EWEATHER_HELPERS_PATH.DS.'translation.helpers.php' );

JLoader::register('eWeatherConfiguration', COM_EWEATHER_CLASSES_PATH.DS.'eweatherconfiguration.php' );
JLoader::register('TWCDailyForecast', COM_EWEATHER_CLASSES_PATH.DS.'twcdailyforecast.php' );
JLoader::register('TWCPartnerLink', COM_EWEATHER_CLASSES_PATH.DS.'twcpartnerlink.php' );


class TWCContent extends JObject
{
    // Header Tag Informations
    public $h_temp = null;
    public $h_distance = null;
    public $h_speed = null;
    public $h_precipitation = null;
    public $h_pressure = null;

    // Location Tag Information
    public $loc_city = null;
    public $loc_id = null;
    public $loc_localtime = null;
    public $loc_longitude = null;
    public $loc_latitude = null;
    public $loc_sunrise = null;
    public $loc_sunset = null;
    public $loc_timezone = null;

    // Current Conditions Tag Information
    public $cc_lastupdate = null;
    public $cc_observatory = null;
    public $cc_temp = null;
    public $cc_windchill = null;
    public $cc_text = null;
    public $cc_icon = null;
    public $cc_windspeed = null;
    public $cc_winddirection = null;
    public $cc_windtext = null;
    public $cc_windgust = null;
    public $cc_barpressure = null;
    public $cc_bartext = null;
    public $cc_humidity = null;
    public $cc_uvindex = null;
    public $cc_uvtext = null;
    public $cc_moonicon = null;
    public $cc_moontext = null;
    public $cc_visibility = null;
    public $cc_dewp =null;

    // Forecast Tag Information
    public $dayf_lastupdate = null;
    public $dayf_forecasts = array();

    // Partner Link Information
    public $partnerLinks = array();
	
	protected $_errorCode = null;
	protected $_errorString = null;
	

	public function loadXML($xmlData)
	{
		try
		{
			// $xmlData is a string, convert it to a DOM document
			$xmlDoc = new DOMDocument();
			$result = $xmlDoc->loadXML($xmlData);

			if (TRUE === $result)
			{
				$docElement = $xmlDoc->documentElement;
				if ($docElement->tagName == 'weather')
				{
    				foreach ($docElement->childNodes as $childNode)
    				{ 
        				$this->_parseNode($childNode); 
    				}
				}
				elseif ($docElement->tagName == 'error')
				{
					//$error = $docElement->childNodes[0]->firstChild->nodeValue."\n";
					//$error.="<br>".JText::_('COM_EWEATHER_ERROR_CODE').": ".$docElement->childNodes[0]->getAttribute('type')."\n";
					$this->_errorCode = $docElement->childNodes[0]->getAttribute('type');
					$this->_errorString = $docElement->childNodes[0]->firstChild->nodeValue;
				}
				else
				{
					$this->_errorCode = null;
					$this->_errorString = JText::_('COM_EWEATHER_UNKNOWN_ERROR');
				}
			}
			else
			{
				$this->_errorCode = null;
				$this->_errorString = "could not parse weather data";
			}
		}
		catch(Exception $exception)
		{
			if (isset($exception))
			{
				$this->_errorCode = $exception->getCode();
				$this->_errorString = $exception->getMessage();
			}
			else
			{
				$this->_errorCode = null;
				$this->_errorString = JText::_('COM_EWEATHER_UNKNOWN_ERROR');
			}
		}
		
	}
	
	public function loadJSON($dayfJSON, $ccJSON)
	{
		$stdClassDayfObject = $this->decodeJSON($dayfJSON);
		if ($this->hasError()) return;
		
		$stdClassCcObject = $this->decodeJSON($ccJSON);
		if ($this->hasError()) return;

		$this->_loadDailyForecastDataFromStdClassObject($stdClassDayfObject);
		$this->_loadCurrentCOnditionsDataFromStdClassObject($stdClassCcObject);
	}
	
	public function toJSON()
	{
		return json_encode($this);
	}

	public function hasError()
	{
		return (isset($this->_errorCode, $this->_errorString));
	}
	
	public function getErrorCode()
	{
		return $this->_errorCode;
	}
	
	public function getErrorString()
	{
		return $this->_errorString;
	}
	
	protected function _parseNode($node)
	{
		$cfg = new eWeatherConfiguration();
		
		switch($node->nodeName)
		{
			case 'head':
    			foreach ($node->childNodes as $childNode)
				{
					if ($childNode->nodeName == "ut") $this->h_temp = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "ud") $this->h_distance = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "us") $this->h_speed = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "ur") $this->h_precipitation = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "up") $this->h_pressure = $childNode->firstChild->nodeValue;
				}
				break;
			
			case 'loc':
				$this->loc_id = $node->getAttribute('id');
    			foreach ($node->childNodes as $childNode)
				{
					if ($childNode->nodeName == "dnam") $this->loc_city = translatePhrase($childNode->firstChild->nodeValue, ', ');
					elseif ($childNode->nodeName == "tm") $this->loc_localtime = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "lat") $this->loc_latitude = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "lon") $this->loc_longitude = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "sunr")
					{
						$this->loc_sunrise = JHTML::_('date', $childNode->firstChild->nodeValue, $cfg->timeFormat, null);
					}
					elseif ($childNode->nodeName == "suns"){
						$this->loc_sunset = JHTML::_('date', $childNode->firstChild->nodeValue, $cfg->timeFormat, null);
					}
					elseif ($childNode->nodeName == "zone") $this->loc_timezone = $childNode->firstChild->nodeValue;
				}
				break;
				
			case 'cc':
    			foreach ($node->childNodes as $childNode)
				{
					if ($childNode->nodeName == "bar")
					{
    					foreach ($childNode->childNodes as $grandchildNode)
						{
							if ($grandchildNode->nodeName == "r") $this->cc_barpressure = $grandchildNode->firstChild->nodeValue;
							elseif ($grandchildNode->nodeName == "d") $this->cc_bartext = $grandchildNode->firstChild->nodeValue;
						}
					}
					elseif ($childNode->nodeName == "uv")
					{
    					foreach ($childNode->childNodes as $grandchildNode)
    					{
							if ($grandchildNode->nodeName == "i") $this->cc_uvindex = $grandchildNode->firstChild->nodeValue;
							elseif ($grandchildNode->nodeName == "t") $this->cc_uvtext = $grandchildNode->firstChild->nodeValue;
						}
					}
					elseif ($childNode->nodeName == "moon")
					{
    					foreach ($childNode->childNodes as $grandchildNode)
    					{
							if ($grandchildNode->nodeName == "icon") $this->cc_moonicon = $grandchildNode->firstChild->nodeValue;
							elseif ($grandchildNode->nodeName == "t") $this->cc_moontext = translatePhrase($grandchildNode->firstChild->nodeValue);
						}
					}
					elseif ($childNode->nodeName == "wind")
					{
    					foreach ($childNode->childNodes as $grandchildNode)
						{
							if ($grandchildNode->nodeName == "s") $this->cc_windspeed = $grandchildNode->firstChild->nodeValue;
							elseif ($grandchildNode->nodeName == "gust") $this->cc_windgust = $grandchildNode->firstChild->nodeValue;
							elseif ($grandchildNode->nodeName == "d") $this->cc_winddirection = $grandchildNode->firstChild->nodeValue;
							elseif ($grandchildNode->nodeName == "t") $this->cc_windtext = $grandchildNode->firstChild->nodeValue;
						}
					}
					elseif ($childNode->nodeName == "lsup")
					{
						$this->cc_lastupdate = JHTML::_('date', $this->cleanUpLastUpdateTime($childNode->firstChild->nodeValue), $cfg->dateFormatLong, null);
					}
					elseif ($childNode->nodeName == "obst") $this->cc_observatory = translatePhrase($childNode->firstChild->nodeValue, ', ');
					elseif ($childNode->nodeName == "tmp") $this->cc_temp = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "flik") $this->cc_windchill = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "t") $this->cc_text = translatePhrase($childNode->firstChild->nodeValue);
					elseif ($childNode->nodeName == "icon") $this->cc_icon = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "hmid") $this->cc_humidity = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "vis") $this->cc_visibility = $childNode->firstChild->nodeValue;
					elseif ($childNode->nodeName == "dewp") $this->cc_dewp = $childNode->firstChild->nodeValue;
				}
				break;

			case 'dayf':
				$dayf_forecasts = array();
    			foreach ($node->childNodes as $childNode)
				{
					if ($childNode->nodeName == "lsup")
					{
						$this->dayf_lastupdate = JHTML::_('date', $this->cleanUpLastUpdateTime($childNode->firstChild->nodeValue), $cfg->dateFormatLong, null);
					}
					elseif ($childNode->nodeName == "day")
					{
						$dailyForecast = new TWCDailyForecast();
						$dailyForecast->loadXMLNode($childNode);
						
						$dayf_forecasts[] = $dailyForecast;
					}
				}
				$this->dayf_forecasts = $dayf_forecasts;
				break;

			case 'lnks':
				$partnerLinks = array();
    			foreach ($node->childNodes as $childNode)
				{
					if ($childNode->nodeName == "link")
					{
						$partnerLink = new TWCPartnerLink();
						$partnerLink->loadXMLNode($childNode);
						
						$partnerLinks[] = $partnerLink;
					}
				}
				$this->partnerLinks = $partnerLinks;
				break;

			default:
				break;
							
		}  // end switch
	}
	
	protected function cleanUpLastUpdateTime($nodeValue)
	{
		$ereg_lsup = str_replace(" Local Time", "", $nodeValue);
		$nodeValue = $ereg_lsup;

		$ereg_lsup = str_replace(" Hawaii ST", " HST", $nodeValue);
		$nodeValue = $ereg_lsup;

		return $nodeValue;
	}

	protected function decodeJSON($jsonString)
	{
		$data = json_decode($jsonString);
		if (null == $data)
		{
			$this->_errorCode = null;
			$this->_errorString = "JSON decode error";
		}
		
		return $data;
	}
	
	protected function _loadDailyForecastDataFromStdClassObject($obj)
	{
		$this->_updateHeaderDataFromStdClassObject($obj);
		$this->_updateLocationDataFromStdClassObject($obj);
		$this->_updateCurrentConditionsDataFromStdClassObject($obj);
		$this->_updateDailyForecastDataFromStdClassObject($obj);
		$this->_updatePartnerLinkDataFromStdClassObject($obj);
	}

	protected function _loadCurrentConditionsDataFromStdClassObject($obj)
	{
		// only if the current conditions data is newer
		if (strtotime($this->cc_lastupdate) < strtotime($obj->cc_lastupdate))
		{
			$this->_updateHeaderDataFromStdClassObject($obj);
			$this->_updateLocationDataFromStdClassObject($obj);
			$this->_updateCurrentConditionsDataFromStdClassObject($obj);
			$this->_updatePartnerLinkDataFromStdClassObject($obj);
		}
	}

	protected function _updateHeaderDataFromStdClassObject($obj)
	{
		$this->h_temp = $obj->h_temp;
		$this->h_distance = $obj->h_distance;
		$this->h_speed = $obj->h_speed;
		$this->h_precipitation = $obj->h_precipitation;
		$this->h_pressure = $obj->h_pressure;
	}

	protected function _updateLocationDataFromStdClassObject($obj)
	{
		$this->loc_city = $obj->loc_city;
		$this->loc_id = $obj->loc_id;
		$this->loc_localtime = $obj->loc_localtime;
		$this->loc_longitude = $obj->loc_longitude;
		$this->loc_latitude = $obj->loc_latitude;
		$this->loc_sunrise = $obj->loc_sunrise;
		$this->loc_sunset = $obj->loc_sunset;
		$this->loc_timezone = $obj->loc_timezone;
	}

	protected function _updateCurrentConditionsDataFromStdClassObject($obj)
	{
		$this->cc_lastupdate = $obj->cc_lastupdate;
		$this->cc_observatory = $obj->cc_observatory;
		$this->cc_temp = $obj->cc_temp;
		$this->cc_windchill = $obj->cc_windchill;
		$this->cc_text = $obj->cc_text;
		$this->cc_icon = $obj->cc_icon;
		$this->cc_windspeed = $obj->cc_windspeed;
		$this->cc_winddirection = $obj->cc_winddirection;
		$this->cc_windtext = $obj->cc_windtext;
		$this->cc_windgust = $obj->cc_windgust;
		$this->cc_barpressure = $obj->cc_barpressure;
		$this->cc_bartext = $obj->cc_bartext;
		$this->cc_humidity = $obj->cc_humidity;
		$this->cc_uvindex = $obj->cc_uvindex;
		$this->cc_uvtext = $obj->cc_uvtext;
		$this->cc_moonicon = $obj->cc_moonicon;
		$this->cc_moontext = $obj->cc_moontext;
		$this->cc_visibility = $obj->cc_visibility;
		$this->cc_dewp = $obj->cc_dewp;
	}

	protected function _updateDailyForecastDataFromStdClassObject($obj)
	{
		$this->dayf_lastupdate = $obj->dayf_lastupdate;
		$this->dayf_forecasts = array();
		
		$dayf_forecasts = $obj->dayf_forecasts;
		foreach ($dayf_forecasts as $dailyForecastObj)
		{
			$dailyForecast = new TWCDailyForecast();
			$dailyForecast->loadStdClassObject($dailyForecastObj);
			
			$this->dayf_forecasts[] = $dailyForecast;
		}
	}
	
	protected function _updatePartnerLinkDataFromStdClassObject($obj)
	{
		$this->partnerLinks = array();
		
		$partnerLinks = $obj->partnerLinks;
		foreach ($partnerLinks as $partnerLinkObj)
		{
			$partnerLink = new TWCPartnerLink();
			$partnerLink->loadStdClassObject($partnerLinkObj);
			
			$this->partnerLinks[] = $partnerLink;
		}
	}
}
