<?php
/**
 * This file contains the model for the main view
 *  for the front-end of the eWeather component.
 *
 * This file is part of eWeather.
 *
 *   eWeather is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   eWeather is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with eWeather.  If not, see <a href="http://www.gnu.org/licenses/">
 *   http://www.gnu.org/licenses/</a>.
 *
 * <b>Modifications:</b><br/>
 * Initial revision by Bob Lavey
 *
 * @package eWeather
 * @subpackage eWeather
 * @copyright 2009 - 2011 Bob Lavey<br/>
 * @license http://www.gnu.org/licenses/gpl.txt GNU/GPL
 */

/* ensure this file is being called by Joomla! */
defined('_JEXEC') or die('Direct Access to this location is not allowed.');

jimport('joomla.application.component.model');

// setup paths
require_once( JPATH_SITE.DS.'components'.DS.'com_eweather'.DS.'helpers'.DS.'defines.php' );
require_once(COM_EWEATHER_HELPERS_PATH.DS.'translation.helpers.php' );

JModel::addIncludePath('COM_EWEATHER_MODELS_PATH');

JLoader::register('eWeatherConfiguration', COM_EWEATHER_CLASSES_PATH.DS.'eweatherconfiguration.php' );
JLoader::register('eWeatherMenuParams', COM_EWEATHER_CLASSES_PATH.DS.'eweathermenuparams.php' );
JLoader::register('eWeatherViewData', COM_EWEATHER_CLASSES_PATH.DS.'eweatherviewdata.php' );
JLoader::register('TheWeatherChannelDataProvider', COM_EWEATHER_CLASSES_PATH.DS.'theweatherchanneldataprovider.php' );
JLoader::register('TWCContent', COM_EWEATHER_CLASSES_PATH.DS.'twccontent.php' );


class eWeatherModelMain extends JModel
{
	protected $_viewData = null;
	protected $_cfg = null;
	protected $_menuParams = null;
	protected $_profileModel = null;
	
	public function __construct($config = array())
	{
		// always call our parent's ctor first
		parent::__construct($config = array());

		$this->_viewData = new eWeatherViewData();
		$this->_profileModel = JModel::getInstance('_Profile', 'eWeatherProfileModel');
	}

	public function getData($locationId = null)
	{
		// reload the config and menu params
		$this->_cfg = new eWeatherConfiguration();
		$this->_menuParams = new eWeatherMenuParams($this->_cfg);
		
		$this->_viewData->errorBackgroundUrl = $this->_getErrorBackgroundUrl();
		$this->_viewData->pageClassSuffix = null;
		$this->_viewData->showPageHeading = false;
		$this->_viewData->pageHeading = null;
		
		// make sure the user has added a menu item
		if (!isset($this->_menuParams->pageclass_sfx))
		{
			$this->_viewData->dataType = 'error';
			$this->_viewData->errorText = JText::_('COM_EWEATHER_ERROR_MUST_ADD_MENUITEM');
		}
		// make sure the user has configured the component
		else if (!isset($this->_cfg->eWeatherVersion))
		{
			$this->_viewData->dataType = 'error';
			$this->_viewData->errorText = JText::_('COM_EWEATHER_ERROR_MUST_CONFIGURE_COMPONENT');
		}
		else if (!isset($this->_cfg->partnerId) || ('' == $this->_cfg->partnerId))
		{
			$this->_viewData->dataType = 'error';
			$this->_viewData->errorText = JText::_('COM_EWEATHER_ERROR_NOPARTNERID');
		}
		else if (!isset($this->_cfg->partnerKey) || ('' == $this->_cfg->partnerKey))
		{
			$this->_viewData->dataType = 'error';
			$this->_viewData->errorText = JText::_('COM_EWEATHER_ERROR_NOPARTNERKEY');
		}
		else
		{
			$this->_viewData->pageClassSuffix = $this->_menuParams->pageclass_sfx;
			$this->_viewData->showPageHeading = $this->_menuParams->show_page_heading;
			$this->_viewData->pageHeading = $this->_menuParams->page_heading;
			
			$this->_viewData->partnerId = $this->_cfg->partnerId;
			$this->_viewData->partnerKey = $this->_cfg->partnerKey;
			
			// if a location code was given in the method call, use it
			if (null != $locationId)
			{
				$this->_viewData->locationId = $locationId;
			}
			// otherwise, figure out the location code to use
			else
			{
				$this->_viewData->locationId = $this->_getLocationId();
			}

			$this->_viewData->units = $this->_getUnits();
			$this->_viewData->forecastCacheTime = $this->_cfg->forecastCacheTime;
			$this->_viewData->ccCacheTime = $this->_cfg->ccCacheTime;
			
			// get the weather data
			$twcContent = $this->_getWeatherData();
			
			if ('error' != $this->_viewData->dataType)
			{
				//
				// make sure the data is valid
				//
				$this->_verifyCurrentConditionsIcon($twcContent->cc_icon);
				$this->_verifyCurrentConditionsMoonIcon($twcContent->cc_moonicon);
				$this->_verifyCurrentConditionsTemperatureAndHumidity($twcContent->cc_temp, $twcContent->cc_windchill, $twcContent->cc_humidity);
				$this->_verifyCurrentConditionsWindAndPressure($twcContent->cc_windtext, $twcContent->cc_windspeed, $twcContent->cc_windgust, $twcContent->cc_winddirection, $twcContent->cc_barpressure);
				$this->_verifyCurrentConditionsUvIndex($twcContent->cc_uvindex, $twcContent->cc_uvtext);
				$this->_verifyForecasts($twcContent->dayf_forecasts);

				$this->_viewData->twcContent = $twcContent;
				$this->_viewData->version = $this->_cfg->eWeatherVersion;
				$this->_viewData->weatherUrl = $this->_getWeatherUrl();
				$this->_viewData->rawWeatherUrl = $this->_getRawWeatherUrl();
				$this->_viewData->profileUrl = $this->_getProfileUrl();
				$this->_viewData->showDetailedForecast = JRequest::getVar('detail', '');
				$this->_viewData->imageUrls = $this->_getImageUrls($twcContent);
				$this->_viewData->forecastDayIcons = $this->_getForecastDayIcons($twcContent);
				$this->_viewData->forecastNightIcons = $this->_getForecastNightIcons($twcContent);
				$this->_viewData->detailedForecastDayIcons = $this->_getDetailedForecastDayIcons($twcContent);
				$this->_viewData->detailedForecastNightIcons = $this->_getDetailedForecastNightIcons($twcContent);
				$this->_viewData->detailedForecastDayWindIcons = $this->_getDetailedForecastDayWindIcons($twcContent);
				$this->_viewData->detailedForecastNightWindIcons = $this->_getDetailedForecastNightWindIcons($twcContent);
				$this->_viewData->formattedCurrentConditionsLastUpdate = $this->_getFormattedCurrentConditionsLastUpdate($twcContent);
				$this->_viewData->formattedForecastLastUpdate = $this->_getFormattedForecastLastUpdate($twcContent);
				$this->_viewData->formattedForecastDates = $this->_getFormattedForecastDates($twcContent);
				$this->_viewData->formattedDetailedForecastDates = $this->_getFormattedDetailedForecastDates($twcContent);
				$this->_viewData->forecastUrl = $this->_getForecastUrl($twcContent);
				$this->_viewData->detailedForecastUrls = $this->_getDetailedForecastUrls($twcContent);
				$this->_viewData->unitUrls = $this->_getUnitUrls();
			}
		}

		return $this->_viewData;
	}

	// set up helper path stubs
	// NOTE these are URLs and not file paths, so don't use DS
	protected function _getWeatherImageLocation()
	{
		return JUri::base(true).COM_EWEATHER_IMAGES_URL;
	}
	
	protected function _getWeatherIconLocation()
	{
		return $this->_getWeatherImageLocation().$this->_cfg->iconStyle.'/';
	}
	
	protected function _getWeatherMoonIconLocation()
	{
		return $this->_getWeatherIconLocation().$this->_cfg->moonIconStyle.'/';
	}
	protected function _getErrorBackgroundUrl()
	{
		return $this->_getWeatherImageLocation().'tbl_error.png';
	}

	protected function _getLocationId()
	{
		// if the user has chosen a location Id, use it
		$locationId = $this->_profileModel->getLocationId();

		// default to the site configuration's default location
		if (!isset($locationId))
		{
			$locationId = $this->_cfg->locationId;
		}

		return $locationId;
	}

	protected function _getUnits()
	{
		// if the user has chosen units, use it
		$units = $this->_profileModel->getUnits();

		// default to the site configuration's default location
		if (!isset($units))
		{
			$units = $this->_cfg->units;
		}

		return $units;
	}

	protected function _getWeatherData()
	{
		$twcProvider = new TheWeatherChannelDataProvider($this->_viewData);
		$twcContent = $twcProvider->getWeatherData();
		$providerError = $twcProvider->getError();
		if (isset($providerError))
		{
			$this->_viewData->dataType = 'error';
			$this->_viewData->errorText = JText::_($providerError);
		}
		
		return $twcContent;
	}
	
	protected function _verifyCurrentConditionsIcon(&$cc_icon)
	{
		if (($cc_icon == null) || ($cc_icon == ""))
		{
			$cc_icon = "na";
		}
		else if (is_numeric($cc_icon))
		{
			$int_icon = intval($cc_icon);

			if (($int_icon < 0) || ($int_icon > 47))
			{
				$cc_icon = "na";
			}
		}
		else
		{
			$cc_icon = "na";
		}

	}

	protected function _verifyCurrentConditionsMoonIcon(&$cc_moonicon)
	{
		if ($cc_moonicon == "")
		{
			$cc_moonicon = null;
		}
		else if (is_numeric($cc_moonicon))
		{
			$int_icon = intval($cc_moonicon);

			if (($int_icon < 0) || ($int_icon > 30))
			{
				$cc_moonicon = null;
			}
		}
		else
		{
			$cc_moonicon = null;
		}

	}

	protected function _verifyCurrentConditionsTemperatureAndHumidity(&$cc_temp, &$cc_windchill, &$cc_humidity)
	{
		if (($cc_temp == null) || ($cc_temp == ""))
		{
			$cc_temp = "N/A";
		}

		if (($cc_windchill == null) || ($cc_windchill == ""))
		{
			$cc_windchill = "N/A";
		}

		if (($cc_humidity == null) || ($cc_humidity == ""))
		{
			$cc_humidity = "N/A";
		}

	}

	protected function _verifyCurrentConditionsWindAndPressure(&$cc_windtext, &$cc_windspeed, &$cc_windgust, &$cc_winddirection, &$cc_barpressure)
	{
		$this->_verifyWindText($cc_windtext);
		$this->_verifyWindSpeed($cc_windspeed);
		$this->_verifyWindGust($cc_windgust);
		$this->_verifyWindDirection($cc_winddirection);
		$this->_verifyPressure($cc_barpressure);
	}

	protected function _verifyCurrentConditionsUvIndex(&$cc_uvindex, &$cc_uvtext)
	{
	}

	protected function _verifyForecasts(&$dayf_forecasts)
	{
		$forecastCount = count($dayf_forecasts);

		for ($x = 0; $x < $forecastCount; $x++)
		{
			$this->_verifyForecastData($dayf_forecasts[$x]);
		}

	}

	protected function _verifyForecastData(&$dayf_forecast)
	{
	}

	protected function _verifyDetailedForecastData(&$windtext, &$windspeed, &$windgust, &$winddirection)
	{
		$this->_verifyWindText($windtext);
		$this->_verifyWindSpeed($windspeed);
		$this->_verifyWindGust($windgust);
		$this->_verifyWindDirection($winddirection);
	}

	protected function _verifyWindText(&$windtext)
	{
		switch ($windtext)
		{
			case 'E':
			case 'ENE':
			case 'ESE':
			case 'N':
			case 'NE':
			case 'NNE':
			case 'NNW':
			case 'NW':
			case 'S':
			case 'SE':
			case 'SSE':
			case 'SSW':
			case 'SW':
			case 'W':
			case 'WNW':
			case 'WSW':
			case 'CALM':
			case 'VAR':
				break;

			default:
				$windtext = null;
				break;
		}
	}

	protected function _verifyWindSpeed(&$windspeed)
	{
		if (!is_numeric($windspeed))
		{
			$windspeed = "N/A";
		}
	}

	protected function _verifyWindGust(&$windgust)
	{
		if (!is_numeric($windgust))
		{
			$windgust = "N/A";
		}
	}

	protected function _verifyWindDirection(&$winddirection)
	{
		if (!is_numeric($winddirection))
		{
			$winddirection = "N/A";
		}
	}

	protected function _verifyPressure(&$barpressure)
	{
		if (!is_numeric($barpressure))
		{
			$barpressure = "N/A";
		}
	}

	protected function _getWeatherUrl()
	{
		return JROUTE::_('index.php?option=com_eweather&amp;Itemid='.$this->_menuParams->menuId.'&amp;task=weather_show');
	}

	protected function _getRawWeatherUrl()
	{
		return 'index.php?option=com_eweather&amp;Itemid='.$this->_menuParams->menuId.'&amp;task=weather_show';
	}

	protected function _getProfileUrl()
	{
		return JROUTE::_('index.php?option=com_eweather&amp;Itemid='.$this->_menuParams->menuId.'&amp;task=profile_show');
	}

	protected function _getImageUrls($twcContent)
	{
		$imageUrls = array();
		$weatherImageLocation = $this->_getWeatherImageLocation();
		$weatherIconLocation = $this->_getWeatherIconLocation();
		$weatherMoonIconLocation = $this->_getWeatherMoonIconLocation();
		
		// NOTE that these are URLs, so don't use DS 
		$imageUrls['weatherChannelLogo'] = $weatherImageLocation.'TWClogo_61px.png';
		$imageUrls['weatherChannelLogoSmall'] = $weatherImageLocation.'TWClogo_31px.png';
		$imageUrls['currentConditions'] = $weatherIconLocation.'large/'.$twcContent->cc_icon.'.png';
		$imageUrls['currentConditionsSmall'] = $weatherIconLocation.'small/'.$twcContent->cc_icon.'.png';
		$imageUrls['moon'] = null;
		if (null != $twcContent->cc_moonicon)
		{
			$imageUrls['moon'] = $weatherMoonIconLocation.$twcContent->cc_moonicon.'.gif';
		}
		$imageUrls['windDir'] = $weatherIconLocation.'winddirs/'.'wind_'.$twcContent->cc_windtext.'.gif';
		$imageUrls['windDirSmall'] = $weatherIconLocation.'winddirs_small/'.'wind_'.$twcContent->cc_windtext.'.gif';
		if ((null == $twcContent->cc_windtext) || ('N/A' == $twcContent->cc_windtext))
		{
			$imageUrls['windDir'] = $weatherIconLocation.'winddirs/wind_nodir.gif';
			$imageUrls['windDirSmall'] = $weatherIconLocation.'winddirs_small/wind_nodir.gif';
		}
		$imageUrls['uvIndexArrow'] = $weatherIconLocation.'arrow.gif';
		$imageUrls['uvIndexKey'] = $weatherIconLocation.'uvkey.gif';
		$imageUrls['weatherDetailOpen'] = $weatherIconLocation.'plus.gif';
		$imageUrls['weatherDetailClose'] = $weatherIconLocation.'minus.gif';

		return $imageUrls;
	}

	protected function _getFormattedCurrentConditionsLastUpdate($twcContent)
	{
		return JHTML::_('date', $twcContent->cc_lastupdate, $this->_cfg->dateFormatLong, null);
	}

	protected function _getFormattedForecastLastUpdate($twcContent)
	{
		return JHTML::_('date', $twcContent->dayf_lastupdate, $this->_cfg->dateFormatLong, null);
	}

	protected function _getFormattedForecastDates($twcContent)
	{
		$forecastDays = $this->_cfg->forecastDays;
		$dateFormat = $this->_cfg->dateFormatShort;

		$formattedForecastDates = array($forecastDays);

		for ($i = 0; $i < $forecastDays; ++$i)
		{
			$formattedForecastDates[$i] = JHTML::_('date', $twcContent->dayf_forecasts[$i]->day_date, $dateFormat, null);
		}

		return $formattedForecastDates;
	}

	protected function _getFormattedDetailedForecastDates($twcContent)
	{
		$forecastDays = $this->_cfg->forecastDays;
		$dateFormat = $this->_cfg->dateFormatDetail;

		$formattedDetailedForecastDates = array($forecastDays);

		for ($i = 0; $i < $forecastDays; ++$i)
		{
			$formattedDetailedForecastDates[$i] = JHTML::_('date', $twcContent->dayf_forecasts[$i]->day_date, $dateFormat, null);
		}

		return $formattedDetailedForecastDates;
	}
	
	protected function _getForecastUrl($twcContent)
	{
		return JROUTE::_('index.php?option=com_eweather&amp;Itemid='.$this->_menuParams->menuId.'&task=weather_show&locid='.$twcContent->loc_id.'#eWeatherForecast');

	}

	protected function _getDetailedForecastUrls($twcContent)
	{
		$forecastDays = $this->_cfg->forecastDays;

		$detailedForecastUrls = array($forecastDays);

		for ($i = 0; $i < $forecastDays; ++$i)
		{
			$detailedForecastUrls[$i] = JROUTE::_('index.php?option=com_eweather&amp;Itemid='.$this->_menuParams->menuId.'&task=weather_show&detail='.$i.'&locid='.$twcContent->loc_id.'#eWeatherForecastDetail');
		}

		return $detailedForecastUrls;
	}

	protected function _isInvalidValue($text)
	{
		return (('na' == $text) || ('NA' == $text) || ('n/a' == $text) || ('N/A' == $text));
	}
	
	protected function _getForecastDayIcons($twcContent)
	{
		$forecastDays = $this->_cfg->forecastDays;
		$forecastDayIcons = array($forecastDays);
		$weatherIconLocation = $this->_getWeatherIconLocation();
		
		for ($i = 0; $i < $forecastDays; ++$i)
		{
			// NOTE that these are URLs, so don't use DS
			if ((0 == $i) && $this->_isInvalidValue($twcContent->dayf_forecasts[$i]->day_d_text))
			{
				$forecastDayIcons[$i] = $weatherIconLocation.'small/na.png';
			}
			else
			{
				$forecastDayIcons[$i] = $weatherIconLocation.'small/'.$twcContent->dayf_forecasts[$i]->day_d_icon.'.png';
			}
		}

		return $forecastDayIcons;
	}

	protected function _getForecastNightIcons($twcContent)
	{
		$forecastDays = $this->_cfg->forecastDays;
		$forecastNightIcons = array($forecastDays);
		$weatherIconLocation = $this->_getWeatherIconLocation();
		
		// NOTE that these are URLs, so don't use DS
		for ($i = 0; $i < $forecastDays; ++$i)
		{
			$forecastNightIcons[$i] = $weatherIconLocation.'small/'.$twcContent->dayf_forecasts[$i]->day_n_icon.'.png';
		}

		return $forecastNightIcons;
	}

	protected function _getDetailedForecastDayIcons($twcContent)
	{
		$forecastDays = $this->_cfg->forecastDays;
		$forecastDayIcons = array($forecastDays);
		$weatherIconLocation = $this->_getWeatherIconLocation();
		
		for ($i = 0; $i < $forecastDays; ++$i)
		{
			// NOTE that these are URLs, so don't use DS
			if ((0 == $i) && $this->_isInvalidValue($twcContent->dayf_forecasts[$i]->day_d_text))
			{
				$forecastDayIcons[$i] = $weatherIconLocation.'large/na.png';
			}
			else
			{
				$forecastDayIcons[$i] = $weatherIconLocation.'large/'.$twcContent->dayf_forecasts[$i]->day_d_icon.'.png';
			}
		}

		return $forecastDayIcons;
	}

	protected function _getDetailedForecastNightIcons($twcContent)
	{
		$forecastDays = $this->_cfg->forecastDays;
		$forecastNightIcons = array($forecastDays);
		$weatherIconLocation = $this->_getWeatherIconLocation();
		
		// NOTE that these are URLs, so don't use DS
		for ($i = 0; $i < $forecastDays; ++$i)
		{
			$forecastNightIcons[$i] = $weatherIconLocation.'large/'.$twcContent->dayf_forecasts[$i]->day_n_icon.'.png';
		}

		return $forecastNightIcons;
	}

	protected function _getDetailedForecastDayWindIcons($twcContent)
	{
		$forecastDays = $this->_cfg->forecastDays;
		$forecastDayWindIcons = array($forecastDays + 1);
		$weatherIconLocation = $this->_getWeatherIconLocation();
		
		// NOTE that these are URLs, so don't use DS
		for ($i = 0; $i < $forecastDays; ++$i)
		{
			$forecastDayWindIcons[$i] = $weatherIconLocation.'winddirs/'.'wind_'.$twcContent->dayf_forecasts[$i]->day_d_windtext.'.gif';
		}

		$forecastDayWindIcons['nodir'] = $weatherIconLocation.'winddirs/wind_nodir.gif';
			
		return $forecastDayWindIcons;
	}

	protected function _getDetailedForecastNightWindIcons($twcContent)
	{
		$forecastDays = $this->_cfg->forecastDays;
		$forecastNightWindIcons = array($forecastDays + 1);
		$weatherIconLocation = $this->_getWeatherIconLocation();
		
		// NOTE that these are URLs, so don't use DS
		for ($i = 0; $i < $forecastDays; ++$i)
		{
			$forecastNightWindIcons[$i] = $weatherIconLocation.'winddirs/'.'wind_'.$twcContent->dayf_forecasts[$i]->day_n_windtext.'.gif';
		}
		
		$forecastNightWindIcons['nodir'] = $weatherIconLocation.'winddirs/wind_nodir.gif';
			
		return $forecastNightWindIcons;
	}

	protected function _getUnitUrls()
	{
		$unitUrls = array();
		
		$unitUrls['s'] = JROUTE::_('index.php?option=com_eweather&amp;Itemid='.$this->_menuParams->menuId.'&task=weather_units');
		$unitUrls['m'] = JROUTE::_('index.php?option=com_eweather&amp;Itemid='.$this->_menuParams->menuId.'&task=weather_unitm');
		
		return $unitUrls;
	}

}
