<?php
/**
 * This file provides a concrete implementation for the
 *  IWeatherDataProvider interface that provides the data
 *  from The Weather Channel.
 *
 * This file is part of eWeather.
 *
 *   eWeather is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   eWeather is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with eWeather.  If not, see <a href="http://www.gnu.org/licenses/">
 *   http://www.gnu.org/licenses/</a>.
 *
 * <b>Modifications:</b><br/>
 * Initial revision by Bob Lavey
 *
 * @package eWeather
 * @subpackage eWeather
 * @copyright 2009 - 2011 Bob Lavey<br/>
 * @license http://www.gnu.org/licenses/gpl.txt GNU/GPL
 */

/* ensure this file is being called by Joomla! */
defined('_JEXEC') or die('Direct Access to this location is not allowed.');

// setup paths
require_once( JPATH_SITE.DS.'components'.DS.'com_eweather'.DS.'helpers'.DS.'defines.php' );

JLoader::register('IWeatherDataProvider', COM_EWEATHER_INTERFACES_PATH.DS.'iweatherdataprovider.php' );
JLoader::register('DomXmlDataProvider', COM_EWEATHER_CLASSES_PATH.DS.'domxmldataprovider.php' );
JLoader::register('CurlXmlDataProvider', COM_EWEATHER_CLASSES_PATH.DS.'curlxmldataprovider.php' );
JLoader::register('TWCContent', COM_EWEATHER_CLASSES_PATH.DS.'twccontent.php' );


class TheWeatherChannelDataProvider implements IWeatherDataProvider
{
	protected $_error = null;
	protected $_viewData = null;
	protected $_xmlDataProvider = null;
	
	public function __construct($viewData)
	{
		$this->_viewData = $viewData;
	}
	
    public function getWeatherData()
    {
		//
		// update the weather data
		//
		$this->_updateDailyForecastData();
		if (isset($this->_error)) return null;

		$this->_updateCurrentConditionsData();
		if (isset($this->_error)) return null;
		
		//
		// retrieve the weather data from the cache and
		//  load it into a TWCContent class object
		//
		$dayfCache = $this->_getCachedDailyForecastData();
		$ccCache = $this->_getCachedCurrentConditionsData();

		$twcContent = new TWCContent();
		$twcContent->loadJSON($dayfCache->data, $ccCache->data);
		
		return $twcContent;
    }
	
	public function getError()
	{
		return $this->_error;
	}
	
	protected function _updateDailyForecastData()
	{
		//
		// get the cached daily forecast data
		//
		$cachedData = $this->_getCachedDailyForecastData();
		if (isset($this->_error)) return;

		//
		// cached data was found
		//
		if (isset($cachedData))
		{
			$cacheId = $cachedData->id;
			$cacheDataExpireTime = time() - $this->_viewData->forecastCacheTime;

			//
			// cached data was found, but it needs to be refreshed
			//
			if ($cachedData->lastupdate <= $cacheDataExpireTime)
			{
				// get the most recent data
				$xmlDataProvider = $this->_getXmlDataProvider();
				$xmlData = $xmlDataProvider->getXmlData($this->_getDailyForecastUrl());

				// update the cache with the most recent data in JSON format
				if (isset($xmlData))
				{
					//echo $xmlData;
					$twcContent = new TWCContent();
					$twcContent->loadXML($xmlData);
					
					if (!$twcContent->hasError())
					{
						$jsonString = $twcContent->toJSON();
					}
					if (isset($jsonString))
					{
						$this->_updateCachedWeatherData($cacheId, $jsonString);
					}
					else
					{
						$this->_setError($twcContent->getErrorCode(), $twcContent->getErrorString(), 'while retrieving Daily Forecasts');
					}
				}
				else
				{
					$this->_setError($xmlDataProvider->getErrorCode(), $xmlDataProvider->getErrorString(), 'while retrieving Daily Forecasts');
				}
			}
		}
		//
		// cached data was not found
		//
		else
		{
			// get the most recent data
			$xmlDataProvider = $this->_getXmlDataProvider();
			$xmlData = $xmlDataProvider->getXmlData($this->_getDailyForecastUrl());

			// add the most recent data to the cache in JSON format
			if (isset($xmlData))
			{
				//echo $xmlData;
				$twcContent = new TWCContent();
				$twcContent->loadXML($xmlData);

				if (!$twcContent->hasError())
				{
					$jsonString = $twcContent->toJSON();
				}
				if (isset($jsonString))
				{
					$this->_addCachedWeatherData('dayf', $this->_viewData->locationId, $this->_viewData->units, $jsonString);
				}
				else
				{
					$this->_setError($twcContent->getErrorCode(), $twcContent->getErrorString(), 'while retrieving Daily Forecasts');
				}
			}
			else
			{
				$this->_setError($xmlDataProvider->getErrorCode(), $xmlDataProvider->getErrorString(), 'while retrieving Daily Forecasts');
			}
		}
	}
	
	protected function _updateCurrentConditionsData()
	{
		//
		// get the cached daily forecast data
		//
		$cachedData = $this->_getCachedCurrentConditionsData();
		if (isset($this->_error)) return;
		
		//
		// cached data was found
		//
		if (isset($cachedData))
		{
			$cacheId = $cachedData->id;
			$cacheDataExpireTime = time() - $this->_viewData->ccCacheTime;

			//
			// cached data was found, but it needs to be refreshed
			//
			if ($cachedData->lastupdate <= $cacheDataExpireTime)
			{
				// get the most recent data
				$xmlDataProvider = $this->_getXmlDataProvider();
				$xmlData = $xmlDataProvider->getXmlData($this->_getCurrentConditionsUrl());

				// update the cache with the most recent data in JSON format
				if (isset($xmlData))
				{
					//echo $xmlData;
					$twcContent = new TWCContent();
					$twcContent->loadXML($xmlData);
					
					if (!$twcContent->hasError())
					{
						$jsonString = $twcContent->toJSON();
					}
					if (isset($jsonString))
					{
						$this->_updateCachedWeatherData($cacheId, $jsonString);
					}
					else
					{
						$this->_setError($twcContent->getErrorCode(), $twcContent->getErrorString(), 'while retrieving Current Conditions');
					}
				}
				else
				{
					$this->_setError($xmlDataProvider->getErrorCode(), $xmlDataProvider->getErrorString(), 'while retrieving Current Conditions');
				}
			}
		}
		//
		// cached data was not found
		//
		else
		{
			// get the most recent data
			$xmlDataProvider = $this->_getXmlDataProvider();
			$xmlData = $xmlDataProvider->getXmlData($this->_getCurrentConditionsUrl());

			// add the most recent data to the cache in JSON format
			if (isset($xmlData))
			{
				//echo $xmlData;
				$twcContent = new TWCContent();
				$twcContent->loadXML($xmlData);
				
				if (!$twcContent->hasError())
				{
					$jsonString = $twcContent->toJSON();
				}
				if (isset($jsonString))
				{
					$this->_addCachedWeatherData('cc', $this->_viewData->locationId, $this->_viewData->units, $jsonString);
				}
				else
				{
					$this->_setError($twcContent->getErrorCode(), $twcContent->getErrorString(), 'while retrieving Current Conditions');
				}
			}
			else
			{
				$this->_setError($xmlDataProvider->getErrorCode(), $xmlDataProvider->getErrorString(), 'while retrieving Current Conditions');
			}
		}
    }

    protected function _getDailyForecastUrl()
    {
		return "http://xoap.weather.com/weather/local/".$this->_viewData->locationId."?cc=*&link=xoap&prod=xoap&dayf=5&unit=".$this->_viewData->units."&par=".$this->_viewData->partnerId."&key=".$this->_viewData->partnerKey;
    }
    
    protected function _getCurrentConditionsUrl()
    {
		return "http://xoap.weather.com/weather/local/".$this->_viewData->locationId."?cc=*&link=xoap&prod=xoap&unit=".$this->_viewData->units."&par=".$this->_viewData->partnerId."&key=".$this->_viewData->partnerKey;
    }
    
	//
	// look at C:\xampp\htdocs\joomla.1.5.20\libraries\domit\php_http_proxy.php
	//  for proxy stuff
	// should be able to use that set of files to get the proxy stuff you need
	//
	// for cUrl proxu stuff, see http://www.cs.sunysb.edu/documentation/curl/index.html
	//  Get a file from a HTTP server that requires user and password, using the same proxy as above: 
	//  curl -u user:passwd -x my-proxy:888 http://www.get.this/ 
	//
	//if ($this->_cfg->useProxy == "1")
	//{
	//	$xmlDoc->setProxyConnection($this->_cfg->proxyHost, '/', $this->_cfg->proxyPort);
	//	if ($this->_cfg->useProxyAuth == "1")
	//	{
	//		$xmlDoc->setProxyAuthorization($this->_cfg->proxyAuthUser, $this->_cfg->proxyAuthPwd);
	//	}
	//}
	
    protected function _getCachedDailyForecastData()
	{
		return $this->_getCachedWeatherData('dayf', $this->_viewData->locationId, $this->_viewData->units);
	}
	
    protected function _getCachedCurrentConditionsData()
	{
		return $this->_getCachedWeatherData('cc', $this->_viewData->locationId, $this->_viewData->units);
	}
	
    protected function _getCachedWeatherData($dataType, $locationId, $units)
	{
		$db = JFactory::getDBO();
		$query = $db->getQuery(true);
		$query->select('id,lastupdate,data');
		$query->from('#__eweather_cache');
		$query->where('dataType='.$db->quote($dataType));
		$query->where('locid='.$db->quote($locationId));
		$query->where('units='.$db->quote($units));
		$db->setQuery($query);
		$cachedData = $db->loadObject();

		return $cachedData;
	}
	
    protected function _addCachedWeatherData($dataType, $locationId, $units, $data)
	{
		$db = JFactory::getDBO();
		$query = $db->getQuery(true);
		$query->insert('#__eweather_cache');
		$query->set("id=''");
		$query->set('lastupdate='.time());
		$query->set('locid='.$db->quote($locationId));
		$query->set('dataType='.$db->quote($dataType));
		$query->set('units='.$db->quote($units));
		$query->set('data='.$db->quote($data));
		$db->setQuery($query);
		$db->query();
	}
	
    protected function _updateCachedWeatherData($cacheId, $data)
	{
		$db = JFactory::getDBO();
		$query = $db->getQuery(true);
		$query->update('#__eweather_cache');
		$query->set('lastupdate='.time());
		$query->set('data='.$db->quote($data));
		$query->where('id='.$cacheId);
		$db->setQuery($query);
		$db->query();
	}

	protected function _setError($errorCode, $errorString, $msg)
	{
		if (isset($errorCode))
		{
			$this->_error = "Error number ".$errorCode.": '".$errorString."' ".$msg;
		}
		else
		{
			$this->_error = "Error: '".$errorString."' ".$msg;
		}
	}
	
	protected function _getXmlDataProvider()
	{
		if (!isset($this->_xmlDataProvider))
		{
			$cfg = new eWeatherConfiguration();
			if ('0' == $cfg->useCUrl)
			{
				$this->_xmlDataProvider = new DomXmlDataProvider();
			}
			else
			{
				$this->_xmlDataProvider = new CurlXmlDataProvider();
			}
		}
		
		return $this->_xmlDataProvider;
	}
}
