<?php
/**
 * This file contains the model for the profile view
 *  for the front-end of the eWeather component.
 *
 * This file is part of eWeather.
 *
 *   eWeather is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   eWeather is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with eWeather.  If not, see <a href="http://www.gnu.org/licenses/">
 *   http://www.gnu.org/licenses/</a>.
 *
 * <b>Modifications:</b><br/>
 * Initial revision by Bob Lavey
 *
 * @package eWeather
 * @subpackage eWeather
 * @copyright 2009 - 2011 Bob Lavey<br/>
 * @license http://www.gnu.org/licenses/gpl.txt GNU/GPL
 */

/* ensure this file is being called by Joomla! */
defined('_JEXEC') or die('Direct Access to this location is not allowed.');

jimport( 'joomla.application.component.modeladmin' );

// setup paths
require_once( JPATH_SITE.DS.'components'.DS.'com_eweather'.DS.'helpers'.DS.'defines.php' );
JTable::addIncludePath(COM_EWEATHER_TABLES_PATH);
JTable::addIncludePath(COM_EWEATHER_ADMIN_TABLES_PATH);

JLoader::register('eWeatherPagination', COM_EWEATHER_CLASSES_PATH.DS.'eweatherpagination.php' );
JLoader::register('eWeatherConfiguration', COM_EWEATHER_CLASSES_PATH.DS.'eweatherconfiguration.php' );


class eWeatherProfileModel_Profile extends JModelAdmin
{
	protected $_where = null;
	protected $_paramsArray = null;
	
	public function __construct($config = array())
	{
		parent::__construct($config);
	}

	/**
     * Method to get the record form.
     *
     * @param       array   $data           Data for the form.
     * @param       boolean $loadData       True if the form is to load its own data (default case), false if not.
     * @return      mixed   A JForm object on success, false on failure
     * @since       1.6
     */
    public function getForm($data = array(), $loadData = true) 
    {
        // Get the form.
        $form = $this->loadForm('com_eweather.site', '_profile', array('control' => 'jform', 'load_data' => $loadData));
		
        if (empty($form)) 
        {
            return false;
        }

        return $form;
    }

    /**
     * Method to get the data that should be injected in the form.
     *
     * @return      mixed   The data for the form.
     * @since       1.6
     */
    protected function loadFormData() 
    {
		$app = JFactory::getApplication();
    	
		// Check for previously entered form data.
    	$data = JRequest::getVar( 'jform', array(), 'post' );
		
        if (empty($data)) 
        {
        	// try to find previous data
			$region = $app->getUserState('com_eweather.userpreferences.filterRegion');
			$country = $app->getUserState('com_eweather.userpreferences.filterCountry');
			$search = $app->getUserState('com_eweather.userpreferences.filterSearch');

			// use default values, if no previous data was found
			$data['region'] = isset($region) ? $region : JText::_('COM_EWEATHER_ADMIN_SELECT_REGION');
			$data['country'] = isset($country) ? $country : JText::_('COM_EWEATHER_ADMIN_SELECT_COUNTRY');
			$data['search'] = isset($search) ?  JString::strtolower($search) : "";
			
			// load user preferences
			$data['locationId'] = $this->getLocationId();
			$data['units'] = $this->getUnits();
        }
        
		// save the published, region, country, and search filter values
		$app->setUserState('com_eweather.userpreferences.filterRegion', $data['region']);
		$app->setUserState('com_eweather.userpreferences.filterCountry', $data['country']);
		$app->setUserState('com_eweather.userpreferences.filterSearch', $data['search']);

        return $data;
    }

	public function clearFilters()
	{
		$app = JFactory::getApplication();
    	
		$app->setUserState('com_eweather.userpreferences.filterRegion', null);
		$app->setUserState('com_eweather.userpreferences.filterCountry', null);
		$app->setUserState('com_eweather.userpreferences.filterSearch', null);
	}

    /**
     * Method to save the given data to the user preferences table
	 * @param data an array of name=>value pairs
     */
    public function save($data)
    {
		$msgData = array();
		$msgData['msg'] = JText::_('COM_EWEATHER_PROFILE_SAVED');
		$msgData['msgType'] = 'message';

		$newParamsArray = array();
		
		// if the user selected a location, use it
		if (array_key_exists('cid', $data))
		{
			$cid = $data['cid'];
			if (count($cid) > 0)
			{
				// use the user-selected id to retrieve the location Id
				$table = JTable::getInstance('eWeatherLocations');
				if ($table && $table->load($cid[0]))
				{
					$newParamsArray['locationId'] = $table->loc_id;
				}
			}
		}
		
		// get the units setting
		if (array_key_exists('units', $data))
		{
			$newParamsArray['units'] = $data['units'];
		}
		
		$newParams = new JRegistry();
		$newParams->loadArray($newParamsArray);
			
		// if the user is logged in, save as a profile record
		$user = JFactory::getUser();
		if (isset($user))
		{
			$userId = $user->get('id');
			if (0 <> $userId)
			{
				$newData = array();
				$newData['params'] = $newParams->toString('JSON');

				// update user's profile, if it exists
				$userProfile = $this->_getUserProfile();
				if (isset($userProfile))
				{
					if (!$userProfile->save($newData))
					{
						$msgData['msg'] = JText::_('COM_EWEATHER_PROFILE_ERROR').': '.$table->getErrors();
						$msgData['msgType'] = 'error';
					}
					else
					{
						$msgData['msg'] = JText::_('COM_EWEATHER_PROFILE_UPDATED');
					}
				}
				// otherwise, add a new record
				else
				{
					$newData['userId'] = $userId;
					$table = JTable::getInstance('eWeatherProfiles');
					if ($table)
					{
						if (!$table->bind($newData))
						{
							$msgData['msg'] = JText::_('COM_EWEATHER_PROFILE_ERROR').': '.$table->getErrors();
							$msgData['msgType'] = 'error';
						}
						else if (!$table->insert())
						{
							$msgData['msg'] = JText::_('COM_EWEATHER_PROFILE_ERROR').': '.$table->getErrors();
							$msgData['msgType'] = 'error';
						}
					}
					else
					{
						$msgData['msg'] = JText::_('COM_EWEATHER_PROFILE_ERROR').': '.$table->getErrors();
						$msgData['msgType'] = 'error';
					}
				}
			}
			// otherwise, user isn't logged in
			// save as cookies
			else
			{
				$this->_saveUserPreferencesToCookies($newParams);
			}
		}
		
		return $msgData;
    }
    
    /**
     * Method to reset the data to defaults
     */
    public function reset()
    {
		// remove the weather_profiles record
		$userProfile = $this->_getUserProfile();
		if (isset($userProfile))
		{
			$userProfile->delete();
		}
		
		// clear cookies by forcing them to expire in the past
		$this->_clearUserPreferenceCookies();
    }
    
	public function getLocationId()
	{
		// check for a locId in the user's preferences
		$locId = $this->_getUserPrefLocationId();
      	
		// check for a locId stored as a cookie
		if (!isset($locId))
		{
			$locId = $this->_getLocIdCookie();
		}
		
      	// if no user profile and no cookie, use the site default
      	if (!isset($locId))
      	{
      		$cfg = new eWeatherConfiguration();
      		$locId = $cfg->locationId;
      	}
		
		return $locId;
	}

	public function getUnits()
	{
		// check for units in the user's preferences
		$units = $this->_getUserPrefUnits();
      	
		// check for units stored as a cookie
		if (!isset($units))
		{
			$units = $this->_getUnitsCookie();
		}
		
      	// if no user profile and no cookie, use the site default
      	if (!isset($units))
      	{
      		$cfg = new eWeatherConfiguration();
      		$units = $cfg->units;
      	}
		
		return $units;
	}

    public function getLocationData()
	{
		$locData = array();
		$locId = null;
		$region = null;
		$country = null;
		$city = null;
		
		$locId = $this->getLocationId();
		
      	// retrieve the region, country, and city using the locId
      	if (isset($locId))
      	{
			$db = JFactory::getDBO();
			$query = $db->getQuery(true);
			$query->select('region,country,city');
			$query->from('#__eweather_locations');
			$query->where('loc_id='.$db->quote($locId));
	        $db->setQuery($query);
	        $row = $db->loadObject();
      		
          	if (isset($row))
          	{
				$region = $row->region;
				$country = $row->country;
				$city = $row->city;
          	}
      	}
      	
		$locData['locId'] = $locId;
		$locData['region'] = $region;
		$locData['country'] = $country;
		$locData['city'] = $city;
      	
		return $locData;
	}

    public function getlocationRows()
	{
		$pagination = $this->getPagination();
			
		$db = JFactory::getDBO();
		$query = $db->getQuery(true);
		
	    // select the fields
	    $query->select('id,region,country,city,loc_id');
        
	    // from the eweather_locations table
       	$query->from('#__eweather_locations');

	    // set the where conditions
		$where = $this->_buildQueryWhere();
		if (strlen($where))
		{
			$query->where($where);
		}
			
	    // set the order conditions
		$query->order('region,country,city');
		
		$locRows = $this->_getList((string) $query, $pagination->limitstart, $pagination->limit);

		return $locRows;
	}

	public function getPagination()
	{
		$config = JFactory::getConfig();
		$pagination = new eWeatherPagination($this->_getTotal(),
							JRequest::getVar('start', 0),
							JRequest::getVar('limit', $config->get('list_limit')));

		return $pagination;
	}

    protected function _buildQueryWhere()
	{
		if (!isset($this->_where))
		{
			// get the published, region, country, and search filter values
			$app = JFactory::getApplication();
			$filterRegion = $app->getUserState('com_eweather.userpreferences.filterRegion');
			$filterCountry = $app->getUserState('com_eweather.userpreferences.filterCountry');
			$search = $app->getUserState('com_eweather.userpreferences.filterSearch');
			
			// prepare to build WHERE clause as an array
			$where = array();
			$db = JFactory::getDBO();

			// check if we are performing a free text search
			if (strlen($search))
			{
				// make string safe for searching
				$search = '%' . $db->getEscaped($search, true). '%';
				$search = $db->Quote($search, false);
				$regionField = $db->nameQuote('region');
				$countryField = $db->nameQuote('country');
				$cityField = $db->nameQuote('city');

				// add search to where array
				$where[] = "(LOWER($regionField) LIKE $search OR LOWER($countryField) LIKE $search OR LOWER($cityField) LIKE $search)";
			}

			// always filter on published = '1'
			$where[] = $db->nameQuote('published') . ' = 1';

			// check if we are filtering based on region
			if (($filterRegion != JText::_('COM_EWEATHER_ADMIN_SELECT_REGION')) && ($filterRegion != ''))
			{
				$where[] = $db->nameQuote('region') . ' = ' . $db->Quote($filterRegion, true);
			}

			// check if we are filtering based on country
			if (($filterCountry != JText::_('COM_EWEATHER_ADMIN_SELECT_COUNTRY')) && ($filterCountry != ''))
			{
				$where[] = $db->nameQuote('country') . ' = ' . $db->Quote($filterCountry, true);
			}

			if (count($where))
			{
				// building from array
				$this->_where = implode(' AND ', $where);
			}
			else
			{
				$this->_where = '';
			}
		}
		
		return $this->_where;
	}

	protected function _getTotal()
	{
		// create a simple query
		$db = JFactory::getDBO();
		$query = $db->getQuery(true);
		
		// select only the id field from the eweather_locations table
		$query->select('id');
		$query->from('#__eweather_locations');
		
		// set the where conditions
		$where = $this->_buildQueryWhere();
		if (strlen($where))
		{
			$query->where($where);
		}
		
		// Load the total.
		$total = (int) $this->_getListCount((string) $query);

		return $total;
	}

	protected function _getUserProfile()
	{
		$userProfile = null;
		
		// get the user's ID and check for an entry
		//  in the eWeatherProfiles table
		$user = JFactory::getUser();
		if (isset($user))
		{
			$userId = $user->get('id');
			if (0 <> $userId)
			{
				$table = JTable::getInstance('eWeatherProfiles');
				if ($table && $table->load($userId))
				{
					$userProfile = $table;
				}
			}
		}
		
		return $userProfile;
	}
	
	protected function _getUserPrefParams()
	{
		if (!isset($this->_paramsArray))
		{
			$this->_paramsArray = array();

			// if we found a row for this user, get the params
			$userProfile = $this->_getUserProfile();
			if (isset($userProfile))
			{
				// convert the JSON string to an array
				$params = new JRegistry();
				$params->loadJSON($userProfile->params);
				$this->_paramsArray = $params->toArray();
			}
		}
		
		return $this->_paramsArray;
	}
	
	protected function _getUserPrefLocationId()
	{
		$locId = null;
		
		$paramsArray = $this->_getUserPrefParams();
		if (array_key_exists('locationId', $paramsArray))
		{
			$locId = $paramsArray['locationId'];
		}
		
		return $locId;
	}
	
	protected function _getUserPrefUnits()
	{
		$units = null;
		
		$paramsArray = $this->_getUserPrefParams();
		if (array_key_exists('units', $paramsArray))
		{
			$units = $paramsArray['units'];
		}
		
		return $units;
	}
	
	protected function _getLocIdCookie()
	{
		$locationId = null;

		if (isset($_COOKIE["defaultWeatherLocation"]))
		{
			$locationId = $_COOKIE["defaultWeatherLocation"];
		}
		
		return $locationId;
	}

	protected function _getUnitsCookie()
	{
		$units = null;
		
		if (isset($_COOKIE["defaultWeatherUnits"]))
		{
			$units = $_COOKIE["defaultWeatherUnits"];
		}
		
		return $units;
	}

	protected function _clearUserPreferenceCookies()
	{
		// clear cookies by forcing them to expire in the past
		if (isset($_COOKIE["defaultWeatherLocation"]))
		{
      		setcookie("defaultWeatherLocation", 0, time()-(3600), '/');
		}

		if (isset($_COOKIE["defaultWeatherUnits"]))
		{
      		setcookie("defaultWeatherUnits", 0, time()-(3600), '/');
		}
	}
	
	protected function _saveUserPreferencesToCookies($newParams)
	{
		$locationId = $newParams->getValue('locationId');
		$units = $newParams->getValue('units');
		
		if (isset($locationId))
		{
			setcookie("defaultWeatherLocation", $locationId, time()+60*60*24*30, '/');
		}
		
		if (isset($units))
		{
			setcookie("defaultWeatherUnits", $units, time()+60*60*24*30, '/');
		}
	}

}
